(function($) {
    'use strict';

    var plutopay = null;
    var paymentForm = null;
    var isProcessing = false;

    function init() {
        if (!window.PlutoPay || !plutopay_params.publishable_key) return;

        plutopay = PlutoPay(plutopay_params.publishable_key);

        $(document.body).on('updated_checkout', onCheckoutUpdated);
        $('form.checkout').on('checkout_place_order_plutopay', onPlaceOrder);
    }

    function onCheckoutUpdated() {
        var container = document.getElementById('plutopay-payment-element');
        if (!container) return;

        // Don't re-mount if already mounted
        if (paymentForm) return;

        // Create payment intent
        $.ajax({
            url: plutopay_params.ajax_url,
            type: 'POST',
            data: {
                action: 'plutopay_create_intent',
                nonce: plutopay_params.nonce,
            },
            success: function(response) {
                if (!response.success) {
                    showError(response.data.message || 'Failed to initialize payment.');
                    return;
                }

                mountPaymentForm(response.data.client_secret, response.data.transaction_id);
            },
            error: function() {
                showError('Connection error. Please refresh the page.');
            }
        });
    }

    function mountPaymentForm(clientSecret, transactionId) {
        var container = document.getElementById('plutopay-payment-element');
        if (!container) return;

        // Store transaction ID
        $('#plutopay_transaction_id').val(transactionId);

        plutopay.mountPaymentForm({
            clientSecret: clientSecret,
            container: '#plutopay-payment-element',
            accountId: plutopay_params.account_id,
            appearance: {
                theme: 'default',
                variables: {
                    colorPrimary: '#0B1437',
                    fontFamily: '-apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif',
                    borderRadius: '6px',
                },
                rules: {
                    '.Input': {
                        border: '1px solid #d1d5db',
                        boxShadow: 'none',
                        padding: '10px 12px',
                        fontSize: '14px',
                    },
                    '.Input:focus': {
                        borderColor: '#0B1437',
                        boxShadow: '0 0 0 2px rgba(11,20,55,0.1)',
                    },
                }
            },
            onReady: function() {
                container.style.minHeight = 'auto';
            },
            onChange: function(event) {
                if (event.complete) {
                    hideError();
                }
            },
        }).then(function(form) {
            paymentForm = form;
        }).catch(function(err) {
            showError('Failed to load payment form: ' + err.message);
        });
    }

    function onPlaceOrder() {
        if (isProcessing) return false;

        if (!paymentForm) {
            showError('Payment form not loaded. Please refresh the page.');
            return false;
        }

        isProcessing = true;
        showProcessing(true);

        paymentForm.submit({
            return_url: plutopay_params.return_url,
        }).then(function(result) {
            if (result.success) {
                // Payment confirmed client-side, submit the WooCommerce form
                var $form = $('form.checkout');
                $form.off('checkout_place_order_plutopay');
                $form.submit();
            } else {
                showError(result.error || 'Payment failed. Please try again.');
                isProcessing = false;
                showProcessing(false);
            }
        }).catch(function(err) {
            showError(err.message || 'An unexpected error occurred.');
            isProcessing = false;
            showProcessing(false);
        });

        return false; // Prevent default form submission
    }

    function showError(message) {
        var $msg = $('#plutopay-payment-message');
        $msg.text(message).show();
    }

    function hideError() {
        $('#plutopay-payment-message').hide();
    }

    function showProcessing(show) {
        if (show) {
            $('form.checkout').addClass('processing').block({
                message: null,
                overlayCSS: { background: '#fff', opacity: 0.6 }
            });
        } else {
            $('form.checkout').removeClass('processing').unblock();
        }
    }

    $(document).ready(init);

})(jQuery);
