<?php
defined('ABSPATH') || exit;

class WC_PlutoPay_Gateway extends WC_Payment_Gateway {

    public function __construct() {
        $this->id                 = 'plutopay';
        $this->icon               = PLUTOPAY_PLUGIN_URL . 'assets/images/plutopay-badge.svg';
        $this->has_fields         = true;
        $this->method_title       = 'PlutoPay';
        $this->method_description = 'Accept credit cards, Apple Pay, Google Pay, Cash App Pay, and more with PlutoPay.';
        $this->supports           = [
            'products',
            'refunds',
        ];

        $this->init_form_fields();
        $this->init_settings();

        $this->title       = $this->get_option('title');
        $this->description = $this->get_option('description');
        $this->enabled     = $this->get_option('enabled');

        add_action('woocommerce_update_options_payment_gateways_' . $this->id, [$this, 'process_admin_options']);
    }

    /**
     * Gateway settings fields
     */
    public function init_form_fields() {
        $this->form_fields = [
            'enabled' => [
                'title'   => 'Enable/Disable',
                'type'    => 'checkbox',
                'label'   => 'Enable PlutoPay',
                'default' => 'no',
            ],
            'title' => [
                'title'       => 'Title',
                'type'        => 'text',
                'description' => 'Payment method title shown at checkout.',
                'default'     => 'Credit Card (PlutoPay)',
                'desc_tip'    => true,
            ],
            'description' => [
                'title'       => 'Description',
                'type'        => 'textarea',
                'description' => 'Payment method description shown at checkout.',
                'default'     => 'Pay securely with your credit card, Apple Pay, Google Pay, or Cash App Pay.',
            ],
            'secret_key' => [
                'title'       => 'Secret Key',
                'type'        => 'password',
                'description' => 'Your PlutoPay secret API key (sk_live_xxx or sk_test_xxx).',
                'desc_tip'    => true,
            ],
            'publishable_key' => [
                'title'       => 'Publishable Key',
                'type'        => 'text',
                'description' => 'Your PlutoPay publishable key (pk_live_xxx or pk_test_xxx).',
                'desc_tip'    => true,
            ],
            'account_id' => [
                'title'       => 'Account ID',
                'type'        => 'text',
                'description' => 'Your PlutoPay Account ID (acct_xxx). Found in Dashboard → Settings.',
                'desc_tip'    => true,
            ],
            'test_mode' => [
                'title'   => 'Test Mode',
                'type'    => 'checkbox',
                'label'   => 'Enable test mode (use test API keys)',
                'default' => 'yes',
            ],
            'checkout_mode' => [
                'title'       => 'Checkout Mode',
                'type'        => 'select',
                'description' => 'How to collect payment at checkout.',
                'default'     => 'embedded',
                'options'     => [
                    'embedded' => 'Embedded form (card fields on checkout page)',
                    'redirect' => 'Redirect (PlutoPay hosted checkout page)',
                ],
            ],
        ];
    }

    /**
     * Get API base URL
     */
    public function get_api_url() {
        return 'https://plutopayus.com/api/v1';
    }

    /**
     * Payment fields shown at checkout
     */
    public function payment_fields() {
        if ($this->description) {
            echo wpautop(wptexturize($this->description));
        }

        if ($this->get_option('checkout_mode') === 'embedded') {
            echo '<div id="plutopay-payment-element" style="padding: 12px 0;"></div>';
            echo '<div id="plutopay-payment-message" style="display:none; color:#dc2626; font-size:13px; margin-top:8px;"></div>';
            echo '<input type="hidden" name="plutopay_transaction_id" id="plutopay_transaction_id" value="">';
        }
    }

    /**
     * Validate payment fields
     */
    public function validate_fields() {
        if ($this->get_option('checkout_mode') === 'embedded') {
            if (empty($_POST['plutopay_transaction_id'])) {
                wc_add_notice('Payment was not completed. Please try again.', 'error');
                return false;
            }
        }
        return true;
    }

    /**
     * Process the payment
     */
    public function process_payment($order_id) {
        $order = wc_get_order($order_id);

        if ($this->get_option('checkout_mode') === 'redirect') {
            return $this->process_redirect_payment($order);
        }

        return $this->process_embedded_payment($order);
    }

    /**
     * Process embedded payment (card fields on page)
     */
    private function process_embedded_payment($order) {
        $transaction_id = sanitize_text_field($_POST['plutopay_transaction_id'] ?? '');

        if (empty($transaction_id)) {
            wc_add_notice('Payment failed. Please try again.', 'error');
            return ['result' => 'failure'];
        }

        // Verify the payment with PlutoPay API
        $response = wp_remote_get($this->get_api_url() . '/transactions/' . $transaction_id, [
            'headers' => [
                'Authorization' => 'Bearer ' . $this->get_option('secret_key'),
                'Accept'        => 'application/json',
            ],
            'timeout' => 30,
        ]);

        if (is_wp_error($response)) {
            wc_add_notice('Payment verification failed.', 'error');
            return ['result' => 'failure'];
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);
        $txn = $body['data'] ?? [];

        if (($txn['status'] ?? '') !== 'succeeded') {
            wc_add_notice('Payment has not been completed.', 'error');
            return ['result' => 'failure'];
        }

        // Payment succeeded
        $order->payment_complete($transaction_id);
        $order->add_order_note(sprintf(
            'PlutoPay payment complete. Transaction: %s | Amount: %s',
            $txn['reference'] ?? $transaction_id,
            ($txn['amount'] ?? 0) / 100
        ));

        // Store PlutoPay transaction ID
        $order->update_meta_data('_plutopay_transaction_id', $transaction_id);
        $order->update_meta_data('_plutopay_reference', $txn['reference'] ?? '');
        $order->save();

        WC()->cart->empty_cart();

        return [
            'result'   => 'success',
            'redirect' => $this->get_return_url($order),
        ];
    }

    /**
     * Process redirect payment (PlutoPay hosted checkout)
     */
    private function process_redirect_payment($order) {
        $amount = intval(round($order->get_total() * 100));
        $currency = strtolower($order->get_currency());

        // Build line items description
        $items = [];
        foreach ($order->get_items() as $item) {
            $items[] = $item->get_name() . ' x' . $item->get_quantity();
        }

        $response = wp_remote_post($this->get_api_url() . '/checkout/sessions', [
            'headers' => [
                'Authorization' => 'Bearer ' . $this->get_option('secret_key'),
                'Content-Type'  => 'application/json',
                'Accept'        => 'application/json',
            ],
            'body' => json_encode([
                'amount'         => $amount,
                'currency'       => $currency,
                'description'    => implode(', ', $items),
                'customer_email' => $order->get_billing_email(),
                'success_url'    => $this->get_return_url($order),
                'cancel_url'     => $order->get_cancel_order_url(),
                'collect_email'  => false,
                'metadata'       => [
                    'order_id'   => $order->get_id(),
                    'order_key'  => $order->get_order_key(),
                    'source'     => 'woocommerce',
                ],
            ]),
            'timeout' => 30,
        ]);

        if (is_wp_error($response)) {
            wc_add_notice('Unable to process payment. Please try again.', 'error');
            return ['result' => 'failure'];
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);

        if (empty($body['url'])) {
            $error = $body['error']['message'] ?? 'Failed to create checkout session';
            wc_add_notice($error, 'error');
            return ['result' => 'failure'];
        }

        // Store checkout reference
        $order->update_meta_data('_plutopay_checkout_ref', $body['reference'] ?? '');
        $order->update_meta_data('_plutopay_transaction_id', $body['transaction_id'] ?? '');
        $order->save();

        $order->add_order_note('Customer redirected to PlutoPay checkout: ' . ($body['reference'] ?? ''));

        return [
            'result'   => 'success',
            'redirect' => $body['url'],
        ];
    }

    /**
     * Process refund
     */
    public function process_refund($order_id, $amount = null, $reason = '') {
        $order = wc_get_order($order_id);
        $transaction_id = $order->get_meta('_plutopay_transaction_id');

        if (empty($transaction_id)) {
            return new WP_Error('plutopay_refund', 'No PlutoPay transaction ID found.');
        }

        $refund_amount = $amount ? intval(round($amount * 100)) : null;

        $body = ['reason' => 'requested_by_customer'];
        if ($refund_amount) {
            $body['amount'] = $refund_amount;
        }

        $response = wp_remote_post($this->get_api_url() . '/transactions/' . $transaction_id . '/refunds', [
            'headers' => [
                'Authorization' => 'Bearer ' . $this->get_option('secret_key'),
                'Content-Type'  => 'application/json',
                'Accept'        => 'application/json',
            ],
            'body'    => json_encode($body),
            'timeout' => 30,
        ]);

        if (is_wp_error($response)) {
            return new WP_Error('plutopay_refund', $response->get_error_message());
        }

        $result = json_decode(wp_remote_retrieve_body($response), true);
        $status = wp_remote_retrieve_response_code($response);

        if ($status !== 200 && $status !== 201) {
            $error = $result['error']['message'] ?? 'Refund failed';
            return new WP_Error('plutopay_refund', $error);
        }

        $order->add_order_note(sprintf(
            'PlutoPay refund processed: $%s. Reason: %s',
            number_format($amount, 2),
            $reason ?: 'N/A'
        ));

        return true;
    }
}
