<?php
/**
 * Plugin Name: PlutoPay for WooCommerce
 * Plugin URI: https://plutopayus.com
 * Description: Accept credit cards, Apple Pay, Google Pay, Cash App Pay, and more with PlutoPay.
 * Version: 1.0.0
 * Author: PlutoPay by Sadeim Inc.
 * Author URI: https://plutopayus.com
 * License: GPL-2.0+
 * Text Domain: plutopay-gateway
 * Requires at least: 5.8
 * Requires PHP: 7.4
 * WC requires at least: 5.0
 * WC tested up to: 8.5
 */

defined('ABSPATH') || exit;

define('PLUTOPAY_VERSION', '1.0.0');
define('PLUTOPAY_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('PLUTOPAY_PLUGIN_URL', plugin_dir_url(__FILE__));

/**
 * Check if WooCommerce is active
 */
function plutopay_check_woocommerce() {
    if (!class_exists('WooCommerce')) {
        add_action('admin_notices', function() {
            echo '<div class="error"><p><strong>PlutoPay Gateway</strong> requires WooCommerce to be installed and active.</p></div>';
        });
        return false;
    }
    return true;
}

/**
 * Initialize the gateway
 */
add_action('plugins_loaded', 'plutopay_init_gateway', 11);

function plutopay_init_gateway() {
    if (!plutopay_check_woocommerce()) return;

    require_once PLUTOPAY_PLUGIN_DIR . 'includes/class-plutopay-gateway.php';

    add_filter('woocommerce_payment_gateways', function($gateways) {
        $gateways[] = 'WC_PlutoPay_Gateway';
        return $gateways;
    });
}

/**
 * Add settings link on plugins page
 */
add_filter('plugin_action_links_' . plugin_basename(__FILE__), function($links) {
    $settings_link = '<a href="admin.php?page=wc-settings&tab=checkout&section=plutopay">Settings</a>';
    array_unshift($links, $settings_link);
    return $links;
});

/**
 * Declare HPOS compatibility
 */
add_action('before_woocommerce_init', function() {
    if (class_exists('\Automattic\WooCommerce\Utilities\FeaturesUtil')) {
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('custom_order_tables', __FILE__, true);
    }
});

/**
 * Enqueue frontend scripts
 */
add_action('wp_enqueue_scripts', function() {
    if (!is_checkout()) return;

    $gateway = new WC_PlutoPay_Gateway();
    if ($gateway->enabled !== 'yes') return;

    wp_enqueue_script(
        'plutopay-js',
        'https://plutopayus.com/js/plutopay.js',
        [],
        PLUTOPAY_VERSION,
        true
    );

    wp_enqueue_script(
        'plutopay-checkout',
        PLUTOPAY_PLUGIN_URL . 'assets/js/checkout.js',
        ['plutopay-js', 'jquery'],
        PLUTOPAY_VERSION,
        true
    );

    wp_localize_script('plutopay-checkout', 'plutopay_params', [
        'publishable_key' => $gateway->get_option('publishable_key'),
        'account_id'      => $gateway->get_option('account_id'),
        'ajax_url'        => admin_url('admin-ajax.php'),
        'nonce'           => wp_create_nonce('plutopay_nonce'),
        'return_url'      => wc_get_checkout_url(),
    ]);

    wp_enqueue_style(
        'plutopay-checkout-style',
        PLUTOPAY_PLUGIN_URL . 'assets/css/checkout.css',
        [],
        PLUTOPAY_VERSION
    );
});

/**
 * AJAX: Create Payment Intent
 */
add_action('wp_ajax_plutopay_create_intent', 'plutopay_ajax_create_intent');
add_action('wp_ajax_nopriv_plutopay_create_intent', 'plutopay_ajax_create_intent');

function plutopay_ajax_create_intent() {
    check_ajax_referer('plutopay_nonce', 'nonce');

    $gateway = new WC_PlutoPay_Gateway();
    $cart_total = WC()->cart->get_total('edit');
    $currency = get_woocommerce_currency();

    $amount = intval(round($cart_total * 100));
    if ($amount < 50) {
        wp_send_json_error(['message' => 'Amount too small']);
        return;
    }

    $response = wp_remote_post($gateway->get_api_url() . '/payment-intents', [
        'headers' => [
            'Authorization' => 'Bearer ' . $gateway->get_option('secret_key'),
            'Content-Type'  => 'application/json',
            'Accept'        => 'application/json',
        ],
        'body' => json_encode([
            'amount'      => $amount,
            'currency'    => strtolower($currency),
            'description' => sprintf('WooCommerce Order - %s', get_bloginfo('name')),
            'metadata'    => [
                'source'    => 'woocommerce',
                'store_url' => get_site_url(),
            ],
        ]),
        'timeout' => 30,
    ]);

    if (is_wp_error($response)) {
        wp_send_json_error(['message' => $response->get_error_message()]);
        return;
    }

    $body = json_decode(wp_remote_retrieve_body($response), true);
    $status = wp_remote_retrieve_response_code($response);

    if ($status !== 201 || empty($body['data']['client_secret'])) {
        $error = $body['error']['message'] ?? 'Failed to create payment';
        wp_send_json_error(['message' => $error]);
        return;
    }

    wp_send_json_success([
        'client_secret'  => $body['data']['client_secret'],
        'transaction_id' => $body['data']['id'],
    ]);
}
